import { z } from './lib';

type KeyRole = 'posting' | 'active' | 'memo'

interface Authority {
  weight_threshold: number
  account_auths?: [string, number][]
  key_auths?: [string, number][]
}

interface Trx {
  ref_block_num: number
  ref_block_prefix: number
  expiration: string
  operations: Operation[]
  extensions: any
}

/**
 * Operation name.
 * Ref: https://gitlab.syncad.com/hive/hive/-/blob/master/libraries/protocol/include/hive/protocol/operations.hpp
 */
type OperationName =
  | 'vote'
  | 'comment'
  | 'transfer'
  | 'transfer_to_vesting'
  | 'withdraw_vesting'
  | 'limit_order_create'
  | 'limit_order_cancel'
  | 'feed_publish'
  | 'convert'
  | 'account_create'
  | 'account_update'
  | 'witness_update'
  | 'account_witness_vote'
  | 'account_witness_proxy'
  | 'pow'
  | 'custom'
  | 'report_over_production'
  | 'delete_comment'
  | 'custom_json'
  | 'comment_options'
  | 'set_withdraw_vesting_route'
  | 'limit_order_create2'
  | 'claim_account'
  | 'create_claimed_account'
  | 'request_account_recovery'
  | 'recover_account'
  | 'change_recovery_account'
  | 'escrow_transfer'
  | 'escrow_dispute'
  | 'escrow_release'
  | 'pow2'
  | 'escrow_approve'
  | 'transfer_to_savings'
  | 'transfer_from_savings'
  | 'cancel_transfer_from_savings'
  | 'custom_binary'
  | 'decline_voting_rights'
  | 'reset_account'
  | 'set_reset_account'
  | 'claim_reward_balance'
  | 'delegate_vesting_shares'
  | 'account_create_with_delegation'
  | 'witness_set_properties'
  | 'account_update2'
  | 'create_proposal'
  | 'update_proposal_votes'
  | 'remove_proposal'
  | 'update_proposal'
  | 'collateralized_convert'
  | 'recurrent_transfer'
/**
 * Virtual operation name.
 */
type VirtualOperationName =
  | 'fill_convert_request'
  | 'author_reward'
  | 'curation_reward'
  | 'comment_reward'
  | 'liquidity_reward'
  | 'interest'
  | 'fill_vesting_withdraw'
  | 'fill_order'
  | 'shutdown_witness'
  | 'fill_transfer_from_savings'
  | 'hardfork'
  | 'comment_payout_update'
  | 'return_vesting_delegation'
  | 'comment_benefactor_reward'
  | 'producer_reward'
  | 'clear_null_account_balance'
  | 'proposal_pay'
  | 'sps_fund'
  | 'hardfork_hive'
  | 'hardfork_hive_restore'
  | 'delayed_voting'
  | 'consolidate_treasury_balance'
  | 'effective_comment_vote'
  | 'ineffective_delete_comment'
  | 'sps_convert'
  | 'expired_account_notification'
  | 'changed_recovery_account'
  | 'transfer_to_vesting_completed'
  | 'pow_reward'
  | 'vesting_shares_split'
  | 'account_created'
  | 'fill_collateralized_convert_request'
  | 'system_warning'
  | 'fill_recurrent_transfer'
  | 'failed_recurrent_transfer'
/**
 * Generic operation.
 */
interface Operation {
  0: OperationName | VirtualOperationName
  1: {
    [key: string]: any
  }
}

declare const schemaDisplayMessage: z.ZodUnion<[z.ZodString, z.ZodObject<{
    title: z.ZodString;
    message: z.ZodOptional<z.ZodString>;
}, "strip", z.ZodTypeAny, {
    title: string;
    message?: string | undefined;
}, {
    title: string;
    message?: string | undefined;
}>]>;
type DisplayMessage = z.infer<typeof schemaDisplayMessage>;

declare const schemaPeakVaultOperation: z.ZodType<[string, {
    [key: string]: any;
}], z.ZodTypeDef, [string, {
    [key: string]: any;
}]>;
type PeakVaultOperation = z.infer<typeof schemaPeakVaultOperation>;

interface PeakVaultResponse {
  success: boolean
  error: string
  message?: string
  account: string
  publicKey?: string
  result: any
}

// Post operation
interface PostJsonMetadata {
  tags?: string[]
  app?: string
  format?: string
  [key: string]: any
}

interface PostOtherOptions {
  max_accepted_payout: string
  percent_hbd: number
  allow_votes: boolean
  allow_curation_rewards: boolean
  extensions:
    | []
    | [[number, { beneficiaries?: { account: string; weight: number }; [key: string]: any }]]
}

declare class PeakVault {
    #private;
    /**
     * Set new RPC node.
     * @param rpc RPC node url.
     */
    setRpc: (rpc: string) => void;
    /**
     * Request user confirmation to sign generic operations using the given key.
     * @param account account with which the user must sign. Use '' to let the user decide which account to use.
     * @param operations operation array that needs to be signed.
     * @param keyRole 'posting', 'active' or 'memo'.
     * @param displayMessage message explaining the operation to the user.
     * @returns
     */
    requestSignOp: (account: string, operations: Operation[] | PeakVaultOperation[], keyRole: KeyRole, displayMessage?: DisplayMessage) => Promise<PeakVaultResponse>;
    /**
     * Request user confirmation to sign a transaction using the given key.
     * @param account account with which the user must sign. Use '' to let the user decide which account to use.
     * @param transaction transaction to sign.
     * @param keyRole 'posting', 'active' or 'memo'.
     * @returns
     */
    requestSignTx: (account: string, transaction: Trx, keyRole: KeyRole) => Promise<PeakVaultResponse>;
    /**
     * Request user confirmation to sign and broadcast generic operations using the given key.
     * @param account account with which the user must sign. Enter '' to let the user decide which account to use.
     * @param operations operation array that needs to be signed.
     * @param keyRole 'posting', 'active' or 'memo'.
     * @param displayMessage message explaining the operation to the user.
     * @returns
     */
    requestBroadcast: (account: string, operations: Operation[], keyRole: KeyRole, displayMessage?: DisplayMessage) => Promise<PeakVaultResponse>;
    /**
     * Ask the user to sign and broadcast a custom JSON operation
     * (i.e. a transaction non-native to the Hive blockchain, like a transaction for a second layer like Splinterlands).
     * @param account account with which the user must sign. Enter '' to let the user decide which account to use.
     * @param id custom JSON id.
     * @param keyRole 'posting', 'active' or 'memo'.
     * @param json custom JSON body.
     * @param displayMessage message explaining the operation to the user.
     */
    requestCustomJson: (account: string, id: string, keyRole: KeyRole, json: {
        [key: string]: any;
    } | string, displayMessage?: DisplayMessage) => Promise<PeakVaultResponse>;
    /**
     * Ask the user to sign and broadcast a transfer operation (i.e. to send money).
     * @param from account that should send the transfer. Enter '' to let the user decide which account to use.
     * @param to account that should receive the transfer.
     * @param amount amount to send.
     * @param currency currency, HIVE or HBD.
     * @param memo message to send along with the transfer. If the first character is # the message will be encrypted.
     * @returns
     */
    requestTransfer: (from: string, to: string, amount: number, currency: "HIVE" | "HBD", memo?: string) => Promise<PeakVaultResponse>;
    /**
     * Ask the user to sign and broadcast a vote operation (i.e. to vote a post).
     * @param voter account that should cast the vote. Enter '' to let the user decide which account to use.
     * @param permlink permlink to the post to vote.
     * @param author author of the post.
     * @param weight vote weight [1-10000].
     * @returns
     */
    requestVote: (voter: string, permlink: string, author: string, weight: number) => Promise<PeakVaultResponse>;
    /**
     * Ask the user to post a blog-post or a comment.
     * @param account author of the post or comment. Enter '' to let the user decide which account to use.
     * @param title title of the comment/post.
     * @param body body of the comment/post.
     * @param parentPermlink for comments must be the permlink to the parent post. For posts it represents the main tag or the community.
     * @param parentAccount for comments must be the account of the author of the parent post. For posts leave empty ('').
     * @param json_metadata metadata, like tags or post format. Try to follow the community rules specified at https://developers.hive.io/apidefinitions/#broadcast_ops_comment
     * @param permlink permlink of the post/comment.
     * @param otherOptions specify advanced options. Check the docs for more information. https://developers.hive.io/apidefinitions/#broadcast_ops_comment_options
     * @returns
     */
    requestPost: (account: string, title: string, body: string, parentPermlink: string, parentAccount: string, json_metadata: PostJsonMetadata, permlink: string, otherOptions?: PostOtherOptions) => Promise<PeakVaultResponse>;
    /**
     * Ask the user to sign a message with one of their keys.
     * @param account account that needs to sign the message.
     * @param keyRole 'posting', 'active' or 'memo'.
     * @param message message to be signed.
     * @param displayMessage message explaining the operation to the user.
     * @returns
     */
    requestSignBuffer: (account: string, keyRole: KeyRole, message: string, displayMessage?: DisplayMessage) => Promise<PeakVaultResponse>;
    /**
     * Returns the list of accounts available in the wallet.
     * @returns
     */
    connect: () => Promise<PeakVaultResponse>;
    /**
     * Verify that the user has the required authority (keyRole) over the account.
     * @param account account to connect. Enter '' to let the user decide which account to use.
     * @param keyRole key role that needs to be verified.
     * @returns
     */
    connect_old: (account: string, keyRole?: KeyRole) => Promise<PeakVaultResponse>;
    /**
     * Ask the user to decode a message (secret) with one of their private keys.
     * @param account account that should decode the message.
     * @param secret message that should be decoded.
     * @param keyRole key role used to encode the message.
     * @returns
     */
    requestDecode: (account: string, secret: string, keyRole?: KeyRole) => Promise<PeakVaultResponse>;
    /**
     * Ask the user (account) to allow accountToAuthorize to sign transactions on their behalf.
     * @param account authorizer.
     * @param accountToAuthorize account to authorize.
     * @param weight weight of accountToAuthorize's signature.
     * @param keyRole key role of the transactions that accountToAuthorize will be able to sign.
     * @returns
     */
    requestAddAccountAuthority: (account: string, accountToAuthorize: string, weight: number, keyRole: "posting" | "active") => Promise<PeakVaultResponse>;
    /**
     * Ask the user (account) to remove the authorization given to accountToRemove to sign transactions on their behalf.
     * @param account authorizer.
     * @param accountToRemove account to remove.
     * @param keyRole key role of the transactions that accountToRemove will not be able to sign anymore.
     * @returns
     */
    requestRemoveAccountAuthority: (account: string, accountToRemove: string, keyRole: "posting" | "active") => Promise<PeakVaultResponse>;
    /**
     * Ask the user (account) to add keyToAuthorize as one of their keys.
     * @param account authorizer.
     * @param keyToAuthorize public key to add.
     * @param weight weight of keyToAuthorize's signature.
     * @param keyRole key role of keyToAuthorize.
     * @returns
     */
    requestAddKeyAuthority: (account: string, keyToAuthorize: string, weight: number, keyRole: "posting" | "active") => Promise<PeakVaultResponse>;
    /**
     * Ask the user (account) to remove the authorization given to keyToRemove to sign transactions on their behalf.
     * @param account authorizer.
     * @param keyToRemove public key to remove.
     * @param keyRole key role of keyToRemove.
     * @returns
     */
    requestRemoveKeyAuthority: (account: string, keyToRemove: string, keyRole: "posting" | "active") => Promise<PeakVaultResponse>;
    /**
     * Vote an account as witness.
     * @param account voter. Enter '' to let the user decide which account to use.
     * @param witness account to vote.
     * @param approve true for positive vote, false for negative vote.
     * @returns
     */
    requestWitnessVote: (account: string, witness: string, approve: boolean) => Promise<PeakVaultResponse>;
    /**
     * Allow account @proxy to vote for witnesses and proposals in @account's name.
     * @param account account that wants to allow proxy to vote for witnesses and proposals in their place. Enter '' to let the user decide which account to use.
     * @param proxy account that should be allowed to vote for witnesses and proposals in @account's name.
     */
    requestGovernanceProxy: (account: string, proxy: string) => Promise<PeakVaultResponse>;
    /**
     * Create a new proposal to get funding from the Decentralized Hive Fund.
     * @param account account that should publish the proposal. Enter '' to let the user decide which account to use.
     * @param fundsReceiver account that should receive the funds.
     * @param dailyPay daily pay in HBD.
     * @param title proposal title.
     * @param permlink permlink to the hive post published by @account explaining the proposal.
     * @param start start date in ISO format.
     * @param end end date in ISO format.
     * @param extensions
     * @returns
     */
    requestCreateProposal: (account: string, fundsReceiver: string, dailyPay: number, title: string, permlink: string, start: string, end: string, extensions?: any) => Promise<PeakVaultResponse>;
    /**
     * Remove proposals by ID.
     * @param creator proposal creator.
     * @param proposalIds proposal IDs. Can be found next to the proposal name at peakd.com/proposals.
     * @param extensions
     * @returns
     */
    requestRemoveProposal: (creator: string, proposalIds: number[], extensions?: any) => Promise<PeakVaultResponse>;
    /**
     * Update proposal votes.
     * @param voter voter.
     * @param proposalIds proposal IDs.Can be found next to the proposal name at peakd.com/proposals.
     * @param approve true to approve the proposals, false to reject them.
     * @param extensions
     * @returns
     */
    requestUpdateProposalVotes: (voter: string, proposalIds: number[], approve: boolean, extensions?: any) => Promise<PeakVaultResponse>;
    /**
     * Sets and overrides the power delegated by delegator to delegatee.
     * @param delegator account that will delegate.
     * @param delegatee account that will receive the delegation.
     * @param amount amount to be delegated.
     * @param unit HP (Hive Power) or VESTS.
     * @returns
     */
    requestDelegation: (delegator: string, delegatee: string, amount: number, unit?: "HP" | "VESTS") => Promise<PeakVaultResponse>;
    /**
     * Stake HIVE to the specified account. Also referred as "power up".
     * @param from account holding the HIVE to be staked. Enter '' to let the user decide which account to use.
     * @param to account to which the HIVE will be staked.
     * @param amount amount of HIVE to stake.
     * @returns
     */
    requestStake: (from: string, to: string, amount: number) => Promise<PeakVaultResponse>;
    /**
     * Unstake HIVE from the account. Also referred as "power down".
     * @param account account.
     * @param amount amount of HIVE to unstake.
     * @returns
     */
    requestUnstake: (account: string, amount: number) => Promise<PeakVaultResponse>;
    /**
     * Ask the user to sign and broadcast a recurrent transfer operation, i.e. to send money at fixed time intervals.
     * The first transfer will be sent immediately.
     * @param from account that should send the transfer. Enter '' to let the user decide which account to use.
     * @param to account that should receive the transfer.
     * @param amount amount to send.
     * @param currency currency, HIVE or HBD.
     * @param memo message to send along with the transfer. If the first character is # the message will be encrypted.
     * @param recurrence time interval (in hours) after which the next transfer is sent. The minimum is 24h.
     * @param executions the number of times the transfer will be executed.
     * @param extensions
     * @returns
     */
    requestRecurrentTransfer: (from: string, to: string, amount: number, currency: "HIVE" | "HBD", memo: string, recurrence: number, executions: number, extensions?: any) => Promise<PeakVaultResponse>;
    /**
     * Ask the user to convert a message using the sender private memo key and the recipient's public memo key.
     * @param account account that encodes the message (sender).
     * @param recipientAccount account that will be able to decode the message.
     * @param message message to be encoded. Must start with #, e.g. '#test' and not 'test'.
     * @returns
     */
    requestEncode: (account: string, recipientAccount: string, message: string) => Promise<PeakVaultResponse>;
    /**
     * Ask the use to encode a message specifying both private and public keys. You can use any combination of key roles,
     * e.g. a public memo key and a private posting key. Remember to use the right key role when decoding.
     * @param account account that encodes the message (sender).
     * @param keyRole key role of the private key. 'posting', 'active' or 'memo'.
     * @param publicKeys public keys that will be able to decode the message.
     * @param message message to be encoded. Must start with #, e.g. '#test' and not 'test'.
     * @returns
     */
    requestEncodeWithKeys: (account: string, keyRole: KeyRole, publicKeys: string[], message: string) => Promise<PeakVaultResponse>;
    /**
     * Ask the user to convert HIVE <--> HBD.
     * @param account account that will perform the conversion.
     * @param amount amount of currency to convert.
     * @param HBD2HIVE if true: HBD --> HIVE, else: HIVE --> HBD.
     * @returns
     */
    requestConversion: (account: string, amount: number, HBD2HIVE: boolean) => Promise<PeakVaultResponse>;
    /**
     * Create a new account that was previously claimed.
     * @param creator account that claimed the account to be created.
     * @param newAccount account to create (claimed).
     * @param ownerAuthority owner authority object for the new account.
     * @param activeAuthority active authority object for the new account.
     * @param postingAuthority posting authority object for the new account.
     * @param publicMemoKey public memo key for the new account.
     * @returns
     */
    requestCreateClaimedAccount: (creator: string, newAccount: string, ownerAuthority: Required<Authority>, activeAuthority: Required<Authority>, postingAuthority: Required<Authority>, publicMemoKey: string) => Promise<PeakVaultResponse>;
    /**
     * Ask the user to choose a Hive account from their contacts.
     * @returns
     */
    requestContact: () => Promise<PeakVaultResponse>;
}

export { PeakVault };
